'use client'

import { useState } from 'react'
import Image from 'next/image'
import Link from 'next/link'
import { services } from '@/lib/data/services'

const categories = ['All', 'Exterior', 'Interior', 'Paint', 'Protection', 'Packages']

export default function ServicesPage() {
  const [selectedCategory, setSelectedCategory] = useState('All')
  const [imageErrors, setImageErrors] = useState<{[key: string]: boolean}>({})
  
  const filteredServices = selectedCategory === 'All' 
    ? services 
    : services.filter(s => s.category === selectedCategory)

  const handleImageError = (serviceId: string) => {
    setImageErrors(prev => ({ ...prev, [serviceId]: true }))
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="container py-20">
        {/* Header */}
        <div className="text-center mb-12">
          <h1 className="text-5xl font-bold mb-4 text-gray-900">
            Our Services
          </h1>
          <p className="text-xl text-gray-600 max-w-3xl mx-auto">
            Professional auto detailing services to keep your vehicle in pristine condition. 
            From basic washes to ceramic coatings, we have you covered.
          </p>
        </div>
        
        {/* Category Filter */}
        <div className="flex flex-wrap justify-center gap-4 mb-12">
          {categories.map(category => (
            <button
              key={category}
              onClick={() => setSelectedCategory(category)}
              className={`px-6 py-2 rounded-full transition-all ${
                selectedCategory === category
                  ? 'bg-red-600 text-white shadow-lg scale-105'
                  : 'bg-white text-gray-700 hover:bg-gray-100 shadow'
              }`}
            >
              {category}
            </button>
          ))}
        </div>
        
        {/* Services Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 gap-8 max-w-6xl mx-auto">
          {filteredServices.map((service) => (
            <div key={service.id} className="card group relative">
              {/* Popular Badge */}
              {service.popular && (
                <div className="absolute -top-3 right-4 z-10">
                  <span className="bg-red-600 text-white px-3 py-1 rounded-full text-xs font-semibold">
                    POPULAR
                  </span>
                </div>
              )}
              
              <div className="flex flex-col h-full">
                {/* Service Header */}
                <div className="mb-4">
                  <h3 className="text-2xl font-bold text-gray-900 mb-2">
                    {service.name}
                  </h3>
                  <div className="flex flex-wrap items-center gap-4 text-sm">
                    <span className="text-red-600 font-bold text-lg">
                      {service.price}
                    </span>
                    <span className="text-gray-500 flex items-center">
                      <svg className="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" />
                      </svg>
                      {service.duration}
                    </span>
                    <span className="bg-gray-100 text-gray-700 px-2 py-1 rounded text-xs">
                      {service.category}
                    </span>
                  </div>
                </div>
                
                {/* Description */}
                <p className="text-gray-600 mb-4">
                  {service.description}
                </p>
                
                {/* Features */}
                <div className="flex-grow">
                  <h4 className="font-semibold text-gray-900 mb-2">Includes:</h4>
                  <ul className="space-y-1">
                    {service.features.slice(0, 4).map((feature, idx) => (
                      <li key={idx} className="flex items-start text-sm">
                        <span className="text-green-500 mr-2 mt-0.5">✓</span>
                        <span className="text-gray-600">{feature}</span>
                      </li>
                    ))}
                    {service.features.length > 4 && (
                      <li className="text-sm text-gray-500 italic ml-6">
                        +{service.features.length - 4} more features
                      </li>
                    )}
                  </ul>
                </div>
                
                {/* CTA Button */}
                <div className="mt-6">
                  <Link 
                    href="/contact" 
                    className="block text-center bg-gray-900 text-white py-3 rounded-lg hover:bg-gray-800 transition-colors font-semibold"
                  >
                    Book This Service
                  </Link>
                </div>
              </div>
            </div>
          ))}
        </div>
        
        {/* Special Offers Section */}
        <div className="mt-20 bg-gradient-to-r from-red-600 to-red-700 rounded-2xl p-12 text-white">
          <div className="max-w-4xl mx-auto text-center">
            <h2 className="text-3xl md:text-4xl font-bold mb-4">
              Special Offers
            </h2>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mt-8">
              <div className="bg-white/10 backdrop-blur rounded-lg p-6">
                <h3 className="text-xl font-bold mb-2">First Time Customer</h3>
                <p className="text-2xl font-bold mb-2">20% OFF</p>
                <p className="text-sm opacity-90">Any detailing service</p>
              </div>
              <div className="bg-white/10 backdrop-blur rounded-lg p-6">
                <h3 className="text-xl font-bold mb-2">Bundle Deal</h3>
                <p className="text-2xl font-bold mb-2">Save $50</p>
                <p className="text-sm opacity-90">Interior + Exterior combo</p>
              </div>
              <div className="bg-white/10 backdrop-blur rounded-lg p-6">
                <h3 className="text-xl font-bold mb-2">Loyalty Program</h3>
                <p className="text-2xl font-bold mb-2">5th Service FREE</p>
                <p className="text-sm opacity-90">Join our rewards program</p>
              </div>
            </div>
            <Link 
              href="/contact" 
              className="inline-block mt-8 bg-white text-red-600 px-8 py-3 rounded-lg font-bold hover:bg-gray-100 transition-colors"
            >
              Claim Your Discount
            </Link>
          </div>
        </div>
        
        {/* FAQ Section */}
        <div className="mt-20">
          <h2 className="text-3xl font-bold text-center mb-12 text-gray-900">
            Frequently Asked Questions
          </h2>
          <div className="max-w-3xl mx-auto space-y-6">
            <div className="bg-white rounded-lg shadow p-6">
              <h3 className="font-bold text-lg mb-2">How long does a typical detail take?</h3>
              <p className="text-gray-600">
                Service times vary from 30 minutes for a basic wash to 8-10 hours for ceramic coating. 
                We provide estimated times for each service and keep you updated throughout the process.
              </p>
            </div>
            <div className="bg-white rounded-lg shadow p-6">
              <h3 className="font-bold text-lg mb-2">Do you offer mobile detailing?</h3>
              <p className="text-gray-600">
                Yes! We offer mobile detailing services in the San Antonio and Roxie areas. 
                Additional travel fees may apply based on distance.
              </p>
            </div>
            <div className="bg-white rounded-lg shadow p-6">
              <h3 className="font-bold text-lg mb-2">What's included in the monthly maintenance plan?</h3>
              <p className="text-gray-600">
                Our monthly plan includes regular washes, interior cleaning, wax applications, 
                priority scheduling, and discounts on additional services. Perfect for keeping your vehicle in top condition year-round.
              </p>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}
